#include "conv.h"


// Sobel kernels
const int Gx[3][3] = {
  {-1, 0, +1},
  {-2, 0, +2},
  {-1, 0, +1}
};
const int Gy[3][3] = {
  {+1, +2, +1},
  { 0,  0,  0},
  {-1, -2, -1}
};


void skip_ws_and_comments(FILE *f) {
    int c;
    // Skip whitespace and comments (# ... endline)
    do {
        c = fgetc(f);
        if (c == '#') { while (c != '\n' && c != EOF) c = fgetc(f); }
    } while (isspace(c));
    ungetc(c, f);//le pointeur est repositionné sur c
}

int read_int(FILE *f) {
  skip_ws_and_comments(f);
  int v = 0, c = fgetc(f);
  if (!isdigit(c)) return -1;
  while (isdigit(c)) {int cASnb = (c - '0'); v = v*10 + cASnb; c = fgetc(f); }
  //c ne désigne plus un chiffre
  ungetc(c, f);//le flux est repositionné sur c et non sur le
	       //caractère suivant
  return v;
}



int load_pgm(const char *path, Image *img) {
    FILE *f = fopen(path, "rb");
    if (!f) return -1;
    char magic[3] = {0};
    if (fread(magic, 1, 2, f) != 2 || magic[0] != 'P' || magic[1] != '5') { fclose(f); return -2; }

    int w = read_int(f);
    int h = read_int(f);
    int maxv = read_int(f);
    printf("load_pgm w:%d:,h:%d,maxv:%d\n",w,h,maxv);
    if (w <= 0 || h <= 0 || maxv <= 0) { fclose(f); return -3; }
    if (maxv > 255) { fclose(f); return -4; } // ne gère pas >8 bits
    
    int c = fgetc(f); // single whitespace after header
    (void)c;

    uint8_t *data = (uint8_t*)malloc((size_t)w*h);
    if (!data) { fclose(f); return -5; }

    size_t rd = fread(data, 1, (size_t) w*h, f);
    fclose(f);
    if (rd != (size_t)w*h) { free(data); return -6; }

    img->w = w; img->h = h; img->data = data;
    return 0;
}

int save_pgm(const char *path, const Image *img) {
    FILE *f = fopen(path, "wb");
    if (!f) return -1;
    fprintf(f, "P5\n%d %d\n255\n", img->w, img->h);
    size_t wr = fwrite(img->data, 1, (size_t)img->w*img->h, f);
    //printf("OK");
    fclose(f);
    return (wr == (size_t)img->w*img->h) ? 0 : -2;
}

int convert_ppm2pgm(const char *source, const char *dest) {
    FILE *f = fopen(source, "rb");
    if (!f) return -1;

    char magic[3] = {0};
    if (fread(magic, 1, 2, f) != 2 || magic[0] != 'P' || magic[1] != '6') {
        fclose(f); return -2;
    }

    int w = read_int(f);
    int h = read_int(f);
    int maxv = read_int(f);
    if (w <= 0 || h <= 0 || maxv <= 0) { fclose(f); return -3; }
    if (maxv > 255) { fclose(f); return -4; } // ne gère pas >8 bits

    int c = fgetc(f); (void)c; // consommer le séparateur

    size_t n = (size_t)w * (size_t)h;
    size_t nbytes = n * 3;
    uint8_t *rgb = (uint8_t*)malloc(nbytes);
    if (!rgb) { fclose(f); return -5; }

    size_t rd = fread(rgb, 1, nbytes, f);
    fclose(f);
    if (rd != nbytes) { free(rgb); return -6; }

    uint8_t *gray = (uint8_t*)malloc(n);
    if (!gray) { free(rgb); return -7; }

    // Rec.601 en entiers: Y ≈ (77R + 150G + 29B + 128) >> 8
    for (size_t i = 0; i < n; ++i) {
        uint8_t R = rgb[3*i], G = rgb[3*i + 1], B = rgb[3*i + 2];
        unsigned y = 77u*R + 150u*G + 29u*B + 128u;
        gray[i] = (uint8_t)(y >> 8);
    }
    free(rgb);

    Image img = { .w = w, .h = h, .data = gray };
    int rc = save_pgm(dest, &img);
    free(gray);
    return rc;
}

// Safe pixel fetch with zero-padding outside
static inline uint8_t at(const Image *img, int x, int y) {
  if (x < 0 || y < 0 || x >= img->w || y >= img->h) return 1;//1 :
							     //noir. On
							     //peut
							     //aussi
							     //prendre
							     //blanc
    return img->data[y*img->w + x];
}

Image *negative(const Image *in) {
    if (!in || !in->data) return NULL; // sécurité

    Image *out = malloc(sizeof(Image));
    //if (!out) return NULL;

    out->w = in->w;
    out->h = in->h;

    size_t n = (size_t)in->w * (size_t)in->h;//size_t pour éviter
					     //dépassement
    out->data = malloc(n);

    for (size_t i = 0; i < n; i++) {
        out->data[i] = (uint8_t)(255 - in->data[i]);
    }

    return out;
}

Image * sobel_edges(const Image *in) {
  Image * out = malloc(sizeof(Image));
  out->w = in->w; out->h = in->h;
  out->data = (uint8_t*)malloc((size_t)in->w*in->h);
    
  for (int y = 0; y < in->h; ++y) {
    for (int x = 0; x < in->w; ++x) {
      int gx = 0, gy = 0;
      // 3x3 convolution centered at (x,y)
      for (int j = -1; j <= 1; ++j) {
	for (int i = -1; i <= 1; ++i) {
	  int p = at(in, x+i, y+j);
	  gx += p * Gx[j+1][i+1];
	  gy += p * Gy[j+1][i+1];
	}
      }
      
      // Fast magnitude approximation and normalization
      int mag = (abs(gx) + abs(gy)) >> 3; // divide by 8
      if (mag > 255) mag = 255;
      
      // THRESHOLD -> binaire (0 ou 255). Pour une carte "grise", commenter les 3 lignes suivantes
      uint8_t e = (mag >= THRESHOLD) ? 255 : 0;
      
      out->data[y*out->w + x] = e; // ou 'mag' si on veut une magnitude en niveaux de gris
        }
    }
  return out;
}

Image * seuil(const Image *src){
  Image * dst = malloc(sizeof(Image));
  dst->w = src->w; dst->h = src->h;
  dst->data = (uint8_t*)malloc((size_t)src->w*src->h);
  for (int i = 0; i < src->w * src->h; ++i) {
    uint8_t e = (src->data[i] >= THRESHOLD) ? 255 : 0;
    dst->data[i] = e;
  }
  return dst;
}


